<?php
/*
Plugin Name: IPValidator Guardian
Plugin URI: https://ipvalidator.ir
Description: سیستم امنیتی جامع وردپرس برای تشخیص و مسدودسازی Proxy، VPN، Tor و VPS با استفاده از API اختصاصی IPValidator.
Version: 2.0.0
Author: IPValidator Team
Author URI: https://ipvalidator.ir
License: GPL v2 or later
Text Domain: ipvg
*/

if (!defined('ABSPATH')) {
    exit; // جلوگیری از دسترسی مستقیم
}

class IPValidator_Guardian
{
    private $table_name;
    private $option_name = 'ipvg_settings';

    public function __construct()
    {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'ipvg_logs';

        // هوک‌های فعال‌سازی/غیرفعال‌سازی
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // هوک‌های ادمین
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'save_settings'));
        add_action('admin_enqueue_scripts', array($this, 'admin_assets'));

        // هوک‌های ماژول‌ها
        add_action('init', array($this, 'firewall_check')); // فایروال (اولویت بالا در اجرا)
        add_filter('registration_errors', array($this, 'check_registration'), 10, 3);
        add_filter('pre_comment_approved', array($this, 'check_comment'), 10, 2);
        add_filter('wp_authenticate_user', array($this, 'check_login'), 10, 2);
    }

    // --- ایجاد جدول لاگ‌ها ---
    public function activate()
    {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $this->table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            ip varchar(45) NOT NULL,
            module varchar(50) NOT NULL,
            reason text NOT NULL,
            created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // تنظیمات پیش‌فرض در اولین فعال‌سازی
        if (!get_option($this->option_name)) {
            $default_settings = array(
                'api_key' => '',
                'whitelist' => '',
                'firewall_message' => 'دسترسی شما به دلایل امنیتی مسدود شده است. (IPValidator Guardian)',
                'block_status_yes' => 0, // پیش‌فرض غیرفعال برای جلوگیری از قفل شدن ناگهانی
                'block_tor' => 0,
                'block_proxy' => 0,
                'block_vpn' => 0,
                'block_vps' => 0,
                'block_server' => 0,
                'enable_firewall' => 0,
                'enable_registration' => 0,
                'enable_comments' => 0,
                'enable_admin' => 0
            );
            add_option($this->option_name, $default_settings);
        }
    }

    public function deactivate()
    {
        // پاکسازی جدول؟ (فعلاً نگه می‌داریم برای تاریخچه)
    }

    // --- منوی ادمین ---
    public function add_admin_menu()
    {
        add_menu_page(
            'IPValidator Guardian',
            'IPValidator',
            'manage_options',
            'ipvg-settings',
            array($this, 'render_settings_page'),
            'dashicons-shield-alt',
            100
        );
    }

    // --- استایل‌ها ---
    public function admin_assets($hook)
    {
        if ('toplevel_page_ipvg-settings' !== $hook) {
            return;
        }
        ?>
        <style>
            :root {
                --ipvg-primary: #6366f1;
                --ipvg-secondary: #8b5cf6;
                --ipvg-bg-light: #f8fafc;
                --ipvg-text-dark: #334155;
                --ipvg-border: #e2e8f0;
            }
            .ipvg-container {
                background: white;
                max-width: 900px;
                margin: 20px auto;
                padding: 30px;
                border-radius: 12px;
                box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
                border-top: 5px solid var(--ipvg-primary);
            }
            .ipvg-header { margin-bottom: 30px; text-align: center; }
            .ipvg-header h2 { color: #1e293b; margin-bottom: 5px; font-size: 2rem; }
            .ipvg-header p { color: var(--ipvg-text-dark); }
            
            .ipvg-section {
                border: 1px solid var(--ipvg-border);
                border-radius: 10px;
                padding: 25px;
                margin-bottom: 25px;
                background: #fff;
                position: relative;
                overflow: hidden;
            }
            /* گرادینت سبک پس‌زمینه سکشن‌ها */
            .ipvg-section::before {
                content: '';
                position: absolute;
                top: 0; left: 0; width: 100%; height: 4px;
                background: linear-gradient(90deg, var(--ipvg-primary), var(--ipvg-secondary));
            }
            
            .ipvg-section h3 {
                margin-top: 0;
                color: #1e293b;
                border-bottom: 1px solid var(--ipvg-border);
                padding-bottom: 10px;
                margin-bottom: 20px;
                font-size: 1.25rem;
            }
            
            .ipvg-form-row { margin-bottom: 20px; }
            .ipvg-label {
                display: block;
                font-weight: 600;
                margin-bottom: 8px;
                color: #475569;
            }
            .ipvg-input-text {
                width: 100%;
                padding: 12px;
                border: 1px solid #cbd5e1;
                border-radius: 6px;
                font-family: monospace;
                font-size: 14px;
                background: #f8fafc;
            }
            .ipvg-input-text:focus {
                border-color: var(--ipvg-primary);
                box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.2);
            }
            .ipvg-textarea { min-height: 100px; }
            
            /* گرید باکس‌ها */
            .ipvg-checkbox-group {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(220px, 1fr));
                gap: 15px;
            }
            .ipvg-checkbox-item {
                background: #f8fafc;
                padding: 12px;
                border-radius: 8px;
                border: 1px solid var(--ipvg-border);
                transition: all 0.2s;
            }
            .ipvg-checkbox-item:hover { border-color: var(--ipvg-secondary); transform: translateY(-2px); }
            .ipvg-checkbox-item label { cursor: pointer; display: block; }
            
            /* استایل خاص باکس "مسدودسازی عمومی" */
            .ipvg-highlight-box {
                background: linear-gradient(135deg, #eff6ff 0%, #f5f3ff 100%);
                border: 2px solid var(--ipvg-primary);
            }

            .ipvg-save-btn {
                background: linear-gradient(135deg, var(--ipvg-primary), var(--ipvg-secondary));
                color: white;
                border: none;
                padding: 15px 40px;
                font-size: 16px;
                border-radius: 8px;
                cursor: pointer;
                font-weight: bold;
                width: 100%;
                box-shadow: 0 4px 6px rgba(99, 102, 241, 0.25);
                transition: transform 0.2s, box-shadow 0.2s;
            }
            .ipvg-save-btn:hover { transform: translateY(-2px); box-shadow: 0 10px 15px rgba(99, 102, 241, 0.3); }
            
            /* جدول لاگ‌ها */
            .ipvg-log-table {
                width: 100%;
                border-collapse: collapse;
                font-size: 14px;
            }
            .ipvg-log-table th, .ipvg-log-table td {
                padding: 12px;
                border-bottom: 1px solid var(--ipvg-border);
                text-align: right;
            }
            .ipvg-log-table th { background: #f1f5f9; color: var(--ipvg-text-dark); }
            .badge-block { background: #fee2e2; color: #b91c1c; padding: 3px 8px; border-radius: 4px; font-size: 11px; font-weight: bold; }
        </style>
        <?php
    }

    // --- ذخیره تنظیمات ---
    public function save_settings()
    {
        if (isset($_POST['ipvg_save_nonce']) && wp_verify_nonce($_POST['ipvg_save_nonce'], 'ipvg_save_settings')) {
            $settings = array(
                'api_key' => sanitize_text_field($_POST['api_key']),
                'whitelist' => sanitize_textarea_field($_POST['whitelist']),
                'firewall_message' => sanitize_textarea_field($_POST['firewall_message']),
                // سیاست‌های مسدودسازی دقیق
                'block_status_yes' => isset($_POST['block_status_yes']) ? 1 : 0,
                'block_tor' => isset($_POST['block_tor']) ? 1 : 0,
                'block_proxy' => isset($_POST['block_proxy']) ? 1 : 0,
                'block_vpn' => isset($_POST['block_vpn']) ? 1 : 0,
                'block_vps' => isset($_POST['block_vps']) ? 1 : 0,
                'block_server' => isset($_POST['block_server']) ? 1 : 0,
                // فعال‌سازی ماژول‌ها
                'enable_firewall' => isset($_POST['enable_firewall']) ? 1 : 0,
                'enable_registration' => isset($_POST['enable_registration']) ? 1 : 0,
                'enable_comments' => isset($_POST['enable_comments']) ? 1 : 0,
                'enable_admin' => isset($_POST['enable_admin']) ? 1 : 0,
            );
            update_option($this->option_name, $settings);
            
            // پاک کردن کش‌های موقت اگر تنظیمات عوض شد (اختیاری)
            // delete_transient('ipvg_cache_clear_flag'); 
        }
    }

    // --- رابط کاربری تنظیمات ---
    public function render_settings_page()
    {
        $settings = get_option($this->option_name);

        // دریافت لاگ‌ها
        global $wpdb;
        $logs = $wpdb->get_results("SELECT * FROM $this->table_name ORDER BY id DESC LIMIT 50");

        ?>
        <div class="ipvg-container">
            <div class="ipvg-header">
                <h2>🛡️ IPValidator Guardian</h2>
                <p>نسخه ۲.۰ - سیاست‌گذاری امنیتی و مسدودسازی هوشمند</p>
            </div>

            <form method="post" action="">
                <?php wp_nonce_field('ipvg_save_settings', 'ipvg_save_nonce'); ?>

                <!-- بخش تنظیمات اتصال -->
                <div class="ipvg-section">
                    <h3>🔑 تنظیمات اتصال API</h3>
                    <div class="ipvg-form-row">
                        <label class="ipvg-label">کلید API (API Key):</label>
                        <input type="text" name="api_key" value="<?php echo esc_attr($settings['api_key']); ?>" class="ipvg-input-text" placeholder="کلید خود را از ipvalidator.ir دریافت کنید">
                    </div>
                    <div class="ipvg-form-row">
                        <label class="ipvg-label">لیست سفید (Whitelist):</label>
                        <textarea name="whitelist" class="ipvg-input-text ipvg-textarea" placeholder="هر IP در یک خط. این IPها هرگز مسدود نخواهند شد."><?php echo esc_textarea($settings['whitelist']); ?></textarea>
                        <small style="color: #64748b;">⚠️ حتماً IP خود را اینجا وارد کنید تا قفل نشوید.</small>
                    </div>
                </div>

                <!-- بخش سیاست مسدودسازی -->
                <div class="ipvg-section">
                    <h3>⚙️ سیاست مسدودسازی پیشرفته</h3>
                    <p class="description" style="margin-bottom: 20px;">مشخص کنید بر اساس کدام پارامترهای خروجی API، دسترسی مسدود شود:</p>
                    
                    <div class="ipvg-checkbox-group">
                        <!-- گزینه جدید: مسدودسازی عمومی -->
                        <div class="ipvg-checkbox-item ipvg-highlight-box">
                            <label style="font-weight:bold; color: #1e40af;">
                                <input type="checkbox" name="block_status_yes" value="1" <?php checked($settings['block_status_yes'], 1); ?>>
                                ⛔ مسدودسازی عمومی<br>
                                <span style="font-size:11px; font-weight:normal; color:#475569;">هرگاه status=yes باشد مسدود شود</span>
                            </label>
                        </div>

                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="block_tor" value="1" <?php checked($settings['block_tor'], 1); ?>> مسدود شبکه Tor</label>
                        </div>
                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="block_proxy" value="1" <?php checked($settings['block_proxy'], 1); ?>> مسدود Proxy</label>
                        </div>
                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="block_vpn" value="1" <?php checked($settings['block_vpn'], 1); ?>> مسدود VPN</label>
                        </div>
                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="block_vps" value="1" <?php checked($settings['block_vps'], 1); ?>> مسدود VPS</label>
                        </div>
                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="block_server" value="1" <?php checked($settings['block_server'], 1); ?>> مسدود Server/Host</label>
                        </div>
                    </div>
                </div>

                <!-- بخش فعال‌سازی ماژول‌ها -->
                <div class="ipvg-section">
                    <h3>🧩 فعال‌سازی ماژول‌های حفاظتی</h3>
                    <p class="description">ماژول‌هایی را که می‌خواهید از سیاست‌های بالا استفاده کنند انتخاب کنید:</p>
                    
                    <div class="ipvg-checkbox-group">
                        <div class="ipvg-checkbox-item" style="border-left: 4px solid var(--ipvg-primary);">
                            <label><input type="checkbox" name="enable_firewall" value="1" <?php checked($settings['enable_firewall'], 1); ?>> 🚫 فایروال کل سایت (Frontend)</label>
                        </div>
                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="enable_registration" value="1" <?php checked($settings['enable_registration'], 1); ?>> 👤 ثبت‌نام کاربران</label>
                        </div>
                        <div class="ipvg-checkbox-item">
                            <label><input type="checkbox" name="enable_comments" value="1" <?php checked($settings['enable_comments'], 1); ?>> 💬 نظرات (تبدیل به اسپم)</label>
                        </div>
                        <div class="ipvg-checkbox-item" style="border-left: 4px solid var(--ipvg-secondary);">
                            <label><input type="checkbox" name="enable_admin" value="1" <?php checked($settings['enable_admin'], 1); ?>> 🔐 ورود به پنل ادمین</label>
                        </div>
                    </div>

                    <div class="ipvg-form-row" style="margin-top: 25px;">
                        <label class="ipvg-label">پیام خطای فایروال:</label>
                        <textarea name="firewall_message" class="ipvg-input-text ipvg-textarea" rows="2"><?php echo esc_textarea($settings['firewall_message']); ?></textarea>
                    </div>
                </div>

                <button type="submit" class="ipvg-save-btn">💾 ذخیره تنظیمات</button>
            </form>

            <!-- بخش لاگ‌ها -->
            <div class="ipvg-section" style="margin-top: 40px;">
                <h3>📋 گزارش مسدودسازی‌های اخیر</h3>
                <table class="ipvg-log-table">
                    <thead>
                        <tr>
                            <th>IP</th>
                            <th>ماژول</th>
                            <th>دلیل مسدودسازی</th>
                            <th>زمان</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($logs) : ?>
                            <?php foreach ($logs as $log) : ?>
                                <tr>
                                    <td style="font-family:monospace; direction:ltr; text-align:right;"><?php echo esc_html($log->ip); ?></td>
                                    <td><span class="badge-block"><?php echo esc_html($log->module); ?></span></td>
                                    <td><?php echo esc_html($log->reason); ?></td>
                                    <td style="direction: ltr; font-size: 12px; color:#64748b;"><?php echo esc_html($log->created_at); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <tr><td colspan="4" style="text-align:center;">هیچ لاگی ثبت نشده است.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php
    }

    // --- هسته مرکزی: بررسی آی‌پی ---
    private function check_ip_status($ip)
    {
        $settings = get_option($this->option_name);
        
        // 1. بررسی لیست سفید (اولویت بالا)
        $whitelist = array_map('trim', explode("\n", $settings['whitelist']));
        if (in_array($ip, $whitelist)) {
            return array('blocked' => false, 'reason' => 'Whitelisted');
        }

        // 2. فراخوانی API (با کش گذاری 5 دقیقه)
        $transient_key = 'ipvg_' . md5($ip);
        $api_data = get_transient($transient_key);

        if (false === $api_data) {
            if (empty($settings['api_key'])) return array('blocked' => false);

            $url = "https://ipvalidator.ir/api/?apikey=" . $settings['api_key'] . "&ip=" . $ip;
            
            // ارسال درخواست
            $response = wp_remote_get($url, array(
                'timeout' => 5,
                'sslverify' => true, // اتصال امن
                'user-agent' => 'WordPress/IPValidator-Guardian'
            ));

            if (is_wp_error($response)) {
                // در صورت خطای API (مثلا قطع اینترنت)، اجازه دسترسی بده تا سایت بالا بیاید
                return array('blocked' => false, 'reason' => 'API Error');
            }

            $body = wp_remote_retrieve_body($response);
            $api_data = json_decode($body, true);
            
            // کش کردن برای 300 ثانیه (5 دقیقه)
            if ($api_data) {
                set_transient($transient_key, $api_data, 300);
            }
        }

        if (!$api_data) return array('blocked' => false);

        // 3. بررسی سیاست‌های مسدودسازی
        $blocked = false;
        $reasons = array();

        // الف) بررسی عمومی (جديد) - مسدود اگر status=yes باشد
        if ($settings['block_status_yes'] && isset($api_data['status']) && $api_data['status'] === 'yes') {
            $blocked = true;
            $reasons[] = 'Status: Yes (Generic Proxy)';
        }

        // ب) بررسی تور
        if ($settings['block_tor'] && (isset($api_data['status_tor']) && $api_data['status_tor'] === 'yes')) {
            $blocked = true;
            $reasons[] = 'Tor Network';
        }

        // ج) بررسی انواع Hosted
        if (isset($api_data['hosted'])) {
            $hosted = $api_data['hosted'];
            if ($settings['block_proxy'] && $hosted === 'proxy') {
                $blocked = true;
                $reasons[] = 'Proxy';
            }
            if ($settings['block_vpn'] && $hosted === 'vpn') {
                $blocked = true;
                $reasons[] = 'VPN';
            }
            if ($settings['block_vps'] && $hosted === 'vps') {
                $blocked = true;
                $reasons[] = 'VPS';
            }
            if ($settings['block_server'] && ($hosted === 'server' || $hosted === 'host')) {
                $blocked = true;
                $reasons[] = 'Server/Host';
            }
        }

        if ($blocked) {
            return array('blocked' => true, 'reason' => implode(', ', $reasons));
        }

        return array('blocked' => false, 'reason' => 'Clean');
    }

    // --- ثبت در لاگ ---
    private function log_attempt($ip, $module, $reason)
    {
        global $wpdb;
        $wpdb->insert(
            $this->table_name,
            array(
                'ip' => $ip,
                'module' => $module,
                'reason' => $reason,
                'created_at' => current_time('mysql')
            )
        );
    }

    // --- ماژول 1: فایروال کل سایت ---
    public function firewall_check()
    {
        $settings = get_option($this->option_name);
        if (empty($settings['enable_firewall'])) return;

        // اگر کاربر وارد شده (Admin) و صفحه ادمین است، چک نکن (برای اینکه بتوانیم تنظیمات را تغییر دهیم)
        if (is_admin() && is_user_logged_in()) return;

        // اگر در بخش لاگین هستیم، صبر کن تا ماژول چک لاگین تصمیم بگیرد
        if (in_array($GLOBALS['pagenow'], array('wp-login.php'))) return;

        $ip = $this->get_user_ip();
        $result = $this->check_ip_status($ip);

        if ($result['blocked']) {
            $this->log_attempt($ip, 'Firewall', $result['reason']);
            wp_die($settings['firewall_message'], 'Access Denied', array('response' => 403));
        }
    }

    // --- ماژول 2: ثبت‌نام ---
    public function check_registration($errors, $sanitized_user_login, $user_email)
    {
        $settings = get_option($this->option_name);
        if (empty($settings['enable_registration'])) return $errors;

        $ip = $this->get_user_ip();
        $result = $this->check_ip_status($ip);

        if ($result['blocked']) {
            $this->log_attempt($ip, 'Registration', $result['reason']);
            $errors->add('blocked_ip', 'ثبت‌نام با این IP مجاز نیست (' . $result['reason'] . ').');
        }

        return $errors;
    }

    // --- ماژول 3: نظرات ---
    public function check_comment($approved, $commentdata)
    {
        $settings = get_option($this->option_name);
        if (empty($settings['enable_comments'])) return $approved;

        // اگر قبلا اسپم شده، رد نشو
        if ($approved === 'spam') return $approved;

        $ip = $commentdata['comment_author_IP'];
        $result = $this->check_ip_status($ip);

        if ($result['blocked']) {
            $this->log_attempt($ip, 'Comment', $result['reason']);
            return 'spam'; // تبدیل به اسپم به جای رد کامل
        }

        return $approved;
    }

    // --- ماژول 4: ورود ادمین ---
    public function check_login($user, $password)
    {
        $settings = get_option($this->option_name);
        if (empty($settings['enable_admin'])) return $user;

        // اگر قبلا ارور داده بود، رد شو
        if (is_wp_error($user)) return $user;

        // فقط اگر کاربر ادمین است بررسی کن
        if (!in_array('administrator', (array) $user->roles)) {
            return $user;
        }

        $ip = $this->get_user_ip();
        $result = $this->check_ip_status($ip);

        if ($result['blocked']) {
            $this->log_attempt($ip, 'Admin Login', $result['reason']);
            return new WP_Error('blocked_ip', 'ورود مدیران با این IP مجاز نیست (' . $result['reason'] . ').');
        }

        return $user;
    }

    // --- دریافت IP کاربر ---
    private function get_user_ip()
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            // گاهی X_FORWARDED_FOR شامل چند IP است، اولی IP اصلی کلاینت است
            $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            $ip = trim($ips[0]);
        } else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return sanitize_text_field($ip);
    }
}

// راه‌اندازی پلاگین
new IPValidator_Guardian();